<?php
/*--------------------------------------------------------------
   GambioProductsRankingOverview.php 2021-08-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;

use Doctrine\DBAL\Connection;
use Gambio\Admin\Modules\Statistics\App\Data\Factory as DataFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory as OverviewFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\MaxEntriesDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined\TimespanDropdown;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\Model\ValueObjects\Data;
use Gambio\Core\Application\ValueObjects\UserPreferences;

class GambioProductsRankingOverview extends GambioWidgetDefinition
{
    private const TYPE = 'gambio.products.ranking.overview';
    
    private const VERSION = '0.1.0';
    
    private const ICON = "data:image/png;base64,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";
    
    private const WIDGET_TITLE = [
        self::LANGUAGE_CODE_GERMAN  => 'Meistverkaufte Produkte',
        self::LANGUAGE_CODE_ENGLISH => 'Best-selling products'
    ];
    
    /**
     * @var DataFactory
     */
    private $dataFactory;
    
    /**
     * @var UserPreferences
     */
    private $userPreferences;
    
    /**
     * @var Connection
     */
    private $connection;
    
    
    /**
     * @inheritDoc
     */
    public function __construct(
        OverviewFactory $overviewFactory,
        DataFactory $dataFactory,
        UserPreferences $userPreferences,
        Connection $connection
    ) {
        $this->dataFactory     = $dataFactory;
        $this->userPreferences = $userPreferences;
        $this->connection      = $connection;
        
        parent::__construct($overviewFactory->createType(self::TYPE),
                            $overviewFactory->createVersion(self::VERSION),
                            $overviewFactory->createIconUsingData(self::ICON),
                            $overviewFactory->useVisualizations()->createTreemapChart(),
                            $overviewFactory->useOptions()->createOptions($overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createTimespanDropdown($overviewFactory),
                                                                          $overviewFactory->useOptions()
                                                                              ->usePredefined()
                                                                              ->createMaxEntriesDropdown($overviewFactory)),
                            $overviewFactory->createTitles($overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_GERMAN),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_GERMAN]),
                                                           $overviewFactory->createTitle($overviewFactory->createLanguageCode(self::LANGUAGE_CODE_ENGLISH),
                                                                                         self::WIDGET_TITLE[self::LANGUAGE_CODE_ENGLISH])));
    }
    
    
    /**
     * @inheritDoc
     */
    public function data(Options $options): Data
    {
        $timespan = $this->dataFactory->useTimespan()->createFromTerm($options->getById(TimespanDropdown::ID)->value());
        $products = $this->connection->createQueryBuilder()
            ->select([
                         'COUNT(products_description.products_id) AS orders',
                         'products_description.products_name AS name'
                     ])
            ->from('orders')
            ->join('orders',
                   'orders_products',
                   'orders_products',
                   'orders.orders_id = orders_products.orders_id')
            ->join('orders_products',
                   'products_description',
                   'products_description',
                   'products_description.products_id = orders_products.products_id')
            ->where('products_description.language_id = :languageId')
            ->andWhere('orders_status NOT IN (:excludedOrderIds)')
            ->andWhere('date_purchased BETWEEN :startDate AND :endDate')
            ->groupBy('products_description.products_id')
            ->orderBy('orders', 'DESC')
            ->setMaxResults((int)$options->getById(MaxEntriesDropdown::ID)->value())
            ->setParameters([
                                ':excludedOrderIds' => implode(',', self::EXCLUDED_ORDER_STATUS_IDS),
                                ':languageId'       => $this->userPreferences->languageId(),
                                ':startDate'        => $timespan->startDate()
                                    ->format(self::QUERY_TIMESPAN_FORMAT_DATE_START),
                                ':endDate'          => $timespan->endDate()
                                    ->format(self::QUERY_TIMESPAN_FORMAT_DATE_END)
                            ])
            ->execute()
            ->fetchAll();
        
        return $this->dataFactory->useMapData()->createMapData($this->dataFactory->useMapData()
                                                                   ->createItems(...array_map(function ($order) {
                                                                       return $this->dataFactory->useMapData()
                                                                           ->createItem($this->dataFactory->useMapData()
                                                                                            ->createItemTitle($order['name']),
                                                                                        $this->dataFactory->useMapData()
                                                                                            ->createItemValue((integer)$order['orders']));
                                                                   },
                                                                       $products)));
    }
}
